/**
 * Queue requests and handle them at your convenience
 *
 * @type {RequestQueue}
 */
H5P.RequestQueue = (function ($, EventDispatcher) {
  /**
   * A queue for requests, will be automatically processed when regaining connection
   *
   * @param {boolean} [options.showToast] Show toast when losing or regaining connection
   * @constructor
   */
  const RequestQueue = function (options) {
    EventDispatcher.call(this);
    this.processingQueue = false;
    options = options || {};

    this.showToast = options.showToast;
    this.itemName = 'requestQueue';
  };

  /**
   * Add request to queue. Only supports posts currently.
   *
   * @param {string} url
   * @param {Object} data
   * @returns {boolean}
   */
  RequestQueue.prototype.add = function (url, data) {
    if (!window.localStorage) {
      return false;
    }

    let storedStatements = this.getStoredRequests();
    if (!storedStatements) {
      storedStatements = [];
    }

    storedStatements.push({
      url: url,
      data: data,
    });

    window.localStorage.setItem(this.itemName, JSON.stringify(storedStatements));

    this.trigger('requestQueued', {
      storedStatements: storedStatements,
      processingQueue: this.processingQueue,
    });
    return true;
  };

  /**
   * Get stored requests
   *
   * @returns {boolean|Array} Stored requests
   */
  RequestQueue.prototype.getStoredRequests = function () {
    if (!window.localStorage) {
      return false;
    }

    const item = window.localStorage.getItem(this.itemName);
    if (!item) {
      return [];
    }

    return JSON.parse(item);
  };

  /**
   * Clear stored requests
   *
   * @returns {boolean} True if the storage was successfully cleared
   */
  RequestQueue.prototype.clearQueue = function () {
    if (!window.localStorage) {
      return false;
    }

    window.localStorage.removeItem(this.itemName);
    return true;
  };

  /**
   * Start processing of requests queue
   *
   * @return {boolean} Returns false if it was not possible to resume processing queue
   */
  RequestQueue.prototype.resumeQueue = function () {
    // Not supported
    if (!H5PIntegration || !window.navigator || !window.localStorage) {
      return false;
    }

    // Already processing
    if (this.processingQueue) {
      return false;
    }

    // Attempt to send queued requests
    const queue = this.getStoredRequests();
    const queueLength = queue.length;

    // Clear storage, failed requests will be re-added
    this.clearQueue();

    // No items left in queue
    if (!queueLength) {
      this.trigger('emptiedQueue', queue);
      return true;
    }

    // Make sure requests are not changed while they're being handled
    this.processingQueue = true;

    // Process queue in original order
    this.processQueue(queue);
    return true
  };

  /**
   * Process first item in the request queue
   *
   * @param {Array} queue Request queue
   */
  RequestQueue.prototype.processQueue = function (queue) {
    if (!queue.length) {
      return;
    }

    this.trigger('processingQueue');

    // Make sure the requests are processed in a FIFO order
    const request = queue.shift();

    const self = this;
    $.post(request.url, request.data)
      .fail(self.onQueuedRequestFail.bind(self, request))
      .always(self.onQueuedRequestProcessed.bind(self, queue))
  };

  /**
   * Request fail handler
   *
   * @param {Object} request
   */
  RequestQueue.prototype.onQueuedRequestFail = function (request) {
    // Queue the failed request again if we're offline
    if (!window.navigator.onLine) {
      this.add(request.url, request.data);
    }
  };

  /**
   * An item in the queue was processed
   *
   * @param {Array} queue Queue that was processed
   */
  RequestQueue.prototype.onQueuedRequestProcessed = function (queue) {
    if (queue.length) {
      this.processQueue(queue);
      return;
    }

    // Finished processing this queue
    this.processingQueue = false;

    // Run empty queue callback with next request queue
    const requestQueue = this.getStoredRequests();
    this.trigger('queueEmptied', requestQueue);
  };

  /**
   * Display toast message on the first content of current page
   *
   * @param {string} msg Message to display
   * @param {boolean} [forceShow] Force override showing the toast
   * @param {Object} [configOverride] Override toast message config
   */
  RequestQueue.prototype.displayToastMessage = function (msg, forceShow, configOverride) {
    if (!this.showToast && !forceShow) {
      return;
    }

    const config = H5P.jQuery.extend(true, {}, {
      position: {
        horizontal : 'centered',
        vertical: 'centered',
        noOverflowX: true,
      }
    }, configOverride);

    H5P.attachToastTo(H5P.jQuery('.h5p-content:first')[0], msg, config);
  };

  return RequestQueue;
})(H5P.jQuery, H5P.EventDispatcher);

/**
 * Request queue for retrying failing requests, will automatically retry them when you come online
 *
 * @type {offlineRequestQueue}
 */
H5P.OfflineRequestQueue = (function (RequestQueue, Dialog) {

  /**
   * Constructor
   *
   * @param {Object} [options] Options for offline request queue
   * @param {Object} [options.instance] The H5P instance which UI components are placed within
   */
  const offlineRequestQueue = function (options) {
    const requestQueue = new RequestQueue();

    // We could handle requests from previous pages here, but instead we throw them away
    requestQueue.clearQueue();

    let startTime = null;
    const retryIntervals = [10, 20, 40, 60, 120, 300, 600];
    let intervalIndex = -1;
    let currentInterval = null;
    let isAttached = false;
    let isShowing = false;
    let isLoading = false;
    const instance = options.instance;

    const offlineDialog = new Dialog({
      headerText: H5P.t('offlineDialogHeader'),
      dialogText: H5P.t('offlineDialogBody'),
      confirmText: H5P.t('offlineDialogRetryButtonLabel'),
      hideCancel: true,
      hideExit: true,
      classes: ['offline'],
      instance: instance,
      skipRestoreFocus: true,
    });

    const dialog = offlineDialog.getElement();

    // Add retry text to body
    const countDownText = document.createElement('div');
    countDownText.classList.add('count-down');
    countDownText.innerHTML = H5P.t('offlineDialogRetryMessage')
      .replace(':num', '<span class="count-down-num">0</span>');

    dialog.querySelector('.h5p-confirmation-dialog-text').appendChild(countDownText);
    const countDownNum = countDownText.querySelector('.count-down-num');

    // Create throbber
    const throbberWrapper = document.createElement('div');
    throbberWrapper.classList.add('throbber-wrapper');
    const throbber = document.createElement('div');
    throbber.classList.add('sending-requests-throbber');
    throbberWrapper.appendChild(throbber);

    requestQueue.on('requestQueued', function (e) {
      // Already processing queue, wait until queue has finished processing before showing dialog
      if (e.data && e.data.processingQueue) {
        return;
      }

      if (!isAttached) {
        const rootContent = document.body.querySelector('.h5p-content');
        if (!rootContent) {
          return;
        }
        offlineDialog.appendTo(rootContent);
        rootContent.appendChild(throbberWrapper);
        isAttached = true;
      }

      startCountDown();
    }.bind(this));

    requestQueue.on('queueEmptied', function (e) {
      if (e.data && e.data.length) {
        // New requests were added while processing queue or requests failed again. Re-queue requests.
        startCountDown(true);
        return;
      }

      // Successfully emptied queue
      clearInterval(currentInterval);
      toggleThrobber(false);
      intervalIndex = -1;
      if (isShowing) {
        offlineDialog.hide();
        isShowing = false;
      }

      requestQueue.displayToastMessage(
        H5P.t('offlineSuccessfulSubmit'),
        true,
        {
          position: {
            vertical: 'top',
            offsetVertical: '100',
          }
        }
      );

    }.bind(this));

    offlineDialog.on('confirmed', function () {
      // Show dialog on next render in case it is being hidden by the 'confirm' button
      isShowing = false;
      setTimeout(function () {
        retryRequests();
      }, 100);
    }.bind(this));

    // Initialize listener for when requests are added to queue
    window.addEventListener('online', function () {
      retryRequests();
    }.bind(this));

    // Listen for queued requests outside the iframe
    window.addEventListener('message', function (event) {
      const isValidQueueEvent = window.parent === event.source
        && event.data.context === 'h5p'
        && event.data.action === 'queueRequest';

      if (!isValidQueueEvent) {
        return;
      }

      this.add(event.data.url, event.data.data);
    }.bind(this));

    /**
     * Toggle throbber visibility
     *
     * @param {boolean} [forceShow] Will force throbber visibility if set
     */
    const toggleThrobber = function (forceShow) {
      isLoading = !isLoading;
      if (forceShow !== undefined) {
        isLoading = forceShow;
      }

      if (isLoading && isShowing) {
        offlineDialog.hide();
        isShowing = false;
      }

      if (isLoading) {
        throbberWrapper.classList.add('show');
      }
      else {
        throbberWrapper.classList.remove('show');
      }
    };
    /**
     * Retries the failed requests
     */
    const retryRequests = function () {
      clearInterval(currentInterval);
      toggleThrobber(true);
      requestQueue.resumeQueue();
    };

    /**
     * Increments retry interval
     */
    const incrementRetryInterval = function () {
      intervalIndex += 1;
      if (intervalIndex >= retryIntervals.length) {
        intervalIndex = retryIntervals.length - 1;
      }
    };

    /**
     * Starts counting down to retrying queued requests.
     *
     * @param forceDelayedShow
     */
    const startCountDown = function (forceDelayedShow) {
      // Already showing, wait for retry
      if (isShowing) {
        return;
      }

      toggleThrobber(false);
      if (!isShowing) {
        if (forceDelayedShow) {
          // Must force delayed show since dialog may be hiding, and confirmation dialog does not
          //  support this.
          setTimeout(function () {
            offlineDialog.show(0);
          }, 100);
        }
        else {
          offlineDialog.show(0);
        }
      }
      isShowing = true;
      startTime = new Date().getTime();
      incrementRetryInterval();
      clearInterval(currentInterval);
      currentInterval = setInterval(updateCountDown, 100);
    };

    /**
     * Updates the count down timer. Retries requests when time expires.
     */
    const updateCountDown = function () {
      const time = new Date().getTime();
      const timeElapsed = Math.floor((time - startTime) / 1000);
      const timeLeft = retryIntervals[intervalIndex] - timeElapsed;
      countDownNum.textContent = timeLeft.toString();

      // Retry interval reached, retry requests
      if (timeLeft <= 0) {
        retryRequests();
      }
    };

    /**
     * Add request to offline request queue. Only supports posts for now.
     *
     * @param {string} url The request url
     * @param {Object} data The request data
     */
    this.add = function (url, data) {
      // Only queue request if it failed because we are offline
      if (window.navigator.onLine) {
        return false;
      }

      requestQueue.add(url, data);
    };
  };

  return offlineRequestQueue;
})(H5P.RequestQueue, H5P.ConfirmationDialog);
;